; From '@' by Dan Muresan
(define-syntax slot@
  (syntax-rules (=)
    ((_ o) o)
    ((_ o slot = v) (slot-set! o 'slot v))
    ((_ o slot . slots) (slot@ (slot-ref o 'slot) . slots))))

(define-syntax define-class
  (syntax-rules ()
    [(_ name () slots)
     (define-class name (<object>) slots) ]
    [(_ name supers slots)
     (define-class name supers slots <class>) ]
    [(_ name () slots meta)
     (define-class name (<object>) slots meta) ]
    [(_ cname (supers ...) (slots ...) meta)
     (define cname (make meta 'name 'cname 'direct-supers (list supers ...) 'direct-slots (list 'slots ...))) ] ) )

(define-syntax define-generic
  (syntax-rules ()
    [(_ n class) (define n (make class 'name 'n))]
    [(_ n) (define n (make-generic 'n))] ) )

(define-syntax (define-method x r c)
  (let ((head (cadr x))
	(body (cddr x))
	(%add-global-method (r 'add-global-method))
	(%make-method (r 'make-method))
	(%lambda (r 'lambda))
	(%list (r 'list))
	(%<top> (r '<top>)))
    (##sys#check-syntax 'define-method head '(symbol . _))
    (##sys#check-syntax 'define-method body '#(_ 1))
    (let gather ([args (##sys#slot head 1)]
		 [specs '()]
		 [vars '()] )
      (if (or (not (pair? args))
	      (memq (car args) '(#!optional #!key #!rest)) )
	  (let ([name (##sys#slot head 0)])
	    `(set! ,name
	       (,%add-global-method
		(##core#global-ref ,name)
		',name
		(,%list ,@(reverse specs))
		;; `call-next-method' not renamed:
		(,%lambda (call-next-method ,@(reverse vars) ,@args) ,@body) ) ) )
	  (let ([arg (##sys#slot args 0)])
	    (gather (##sys#slot args 1)
		    (cons (if (pair? arg) (cadr arg) %<top>) specs)
		    (cons (if (pair? arg) (car arg) arg) vars) ) ) ) ) ) )

;; For system use in extending the set of "builtin" classes.

(define-for-syntax (##tinyclos#make-classname-symbol str)
  (string->symbol (string-append "<" (##sys#strip-syntax str) ">")) )

(define-syntax (define-primitive-class x r c)
  (let ((name (cadr x))
	(pred (caddr x))
	(sclasses (cdddr x)))
    `(,(r 'define) ,(##tinyclos#make-classname-symbol name)
      (,(r 'new-primitive-class) ,name ,pred ,@sclasses)) ))

(define-syntax (define-structure-class x r c)
  (let ((name (cadr x))
	(tag (caddr x)))
    `(,(r 'define) ,(##tinyclos#make-classname-symbol name)
      (,(r 'new-structure-class) name (,(r 'quote) ,tag)) )))

(define-syntax (define-tagged-pointer-class x r c)
  (let ((name (cadr x))
	(pred (caddr x)))
    `(,(r 'define) ,(##tinyclos#make-classname-symbol name)
      (,(r 'new-tagged-pointer-class) name (,(r 'quote) ,pred)) )))

(define-syntax (define-extended-procedure-class x r c)
  (let ((name (cadr x))
	(pred (caddr x)))
    `(,(r 'define) ,(##tinyclos#make-classname-symbol name)
      (,(r 'new-extended-procedure-class) name (,(r 'quote) ,pred)) )))

;
; A very simple CLOS-like language, embedded in Scheme, with a simple
; MOP.  The features of the default base language are:
;
;   * Classes, with instance slots, but no slot options.
;   * Multiple-inheritance.
;   * Generic functions with multi-methods and class specializers only.
;   * Primary methods and call-next-method; no other method combination.
;   * Uses Scheme's lexical scoping facilities as the class and generic
;     function naming mechanism.  Another way of saying this is that
;     class, generic function and methods are first-class (meta)objects.
;
; While the MOP is simple, it is essentially equal in power to both MOPs
; in AMOP.  This implementation is not at all optimized, but the MOP is
; designed so that it can be optimized.  In fact, this MOP allows better
; optimization of slot access extenstions than those in AMOP.
;
;
;
; In addition to calling a generic, the entry points to the default base
; language are:
;
;   (MAKE-CLASS list-of-superclasses list-of-slot-names)
;   (MAKE-GENERIC)
;   (MAKE-METHOD list-of-specializers procedure)
;   (ADD-METHOD generic method)
;
;   (MAKE class . initargs)
;   (INITIALIZE instance initargs)            ;Add methods to this,
;                                             ;don't call it directly.
;
;   (SLOT-REF  object slot-name)
;   (SLOT-SET! object slot-name new-value)
;
;
; So, for example, one might do (but only within this file! 'getl' is not exported):
;
;   (define <position> (make-class (list <object>) (list 'x 'y)))
;   (add-method initialize
;       (make-method (list <position>)
;         (lambda (call-next-method pos initargs)
;           (for-each (lambda (initarg-name slot-name)
;                       (slot-set! pos slot-name (getl initargs initarg-name 0)))
;                     '(x y)
;                     '(x y)))))
;
;   (set! p1 (make <position> 'x 1 'y 3))
;
;
;
; NOTE!  Do not use EQUAL? to compare objects!  Use EQ? or some hand
;        written procedure.  Objects have a pointer to their class,
;        and classes are circular structures, and ...
;
;
;
; The introspective part of the MOP looks like the following.  Note that
; these are ordinary procedures, not generics.
;
;   CLASS-OF
;
;   CLASS-DIRECT-SUPERS
;   CLASS-DIRECT-SLOTS
;   CLASS-CPL
;   CLASS-SLOTS
;
;   GENERIC-METHODS
;
;   METHOD-SPECIALIZERS
;   METHOD-PROCEDURE
;
;
; The intercessory protocol looks like (generics in uppercase):
;
;   make
;     ALLOCATE-INSTANCE
;     INITIALIZE                   (really a base-level generic)
;
;   class initialization
;     COMPUTE-CPL
;     COMPUTE-SLOTS
;     COMPUTE-GETTER-AND-SETTER
;
;   add-method                     (Notice this is not a generic!)
;     COMPUTE-APPLY-GENERIC
;       COMPUTE-METHODS
;         COMPUTE-METHOD-MORE-SPECIFIC?
;       COMPUTE-APPLY-METHODS
;

;;; Aliases for sys stuff

(define-inline (%car p) (##sys#slot p 0))
(define-inline (%cdr p) (##sys#slot p 1))
(define-inline (%cadr p) (%car (%cdr p)))
(define-inline (%cddr p) (%cdr (%cdr p)))
(define-inline (%set-car! p y) (##sys#setslot p 0 y))
(define-inline (%set-cdr! p y) (##sys#setslot p 1 y))

(define-inline (%string-length s) (##sys#size s))

(define-inline (%vector-ref v i) (##sys#slot v i))
(define-inline (%vector-set! v i x) (##sys#setslot v i x))
(define-inline (%vector-length v) (##sys#size v))

(define-syntax (%structure? x r c)
  (let ((?x (cadr x))
	(?t (cddr x)))
    (if (null? ?t)
	`(##sys#generic-structure? ,?x)
	`(##sys#structure? ,?x ,(car ?t)))))

(define-inline (%structure-ref r i) (##sys#slot r i))
(define-inline (%structure-set! r i x) (##sys#setslot r i x))
(define-inline (%structure-length r) (##sys#size r))
(define-inline (%structure-tag r) (##sys#slot r 0))

(define-inline (%closure-ref c i) (##sys#slot c i))
(define-inline (%closure-length c) (##sys#size c))

(define-inline (%tagged-pointer-data p) (##sys#slot p 1))

(define-inline (%null-pointer? p) (##sys#null-pointer? p))

(define-inline (%blob? x) (##sys#bytevector? x))

(define-inline (%immediate-value? x) (##sys#immediate? x))

(define-inline (%undefined? x) (##core#inline "C_undefinedp" x))

(define-inline (%unbound? x) (eq? x (##sys#slot '##sys#arbitrary-unbound-symbol 0)))

(define-inline (%pointer? x) (##core#inline "C_pointerp" x))

(define-inline (%tagged-pointer? x) (##core#inline "C_taggedpointerp" x))

(define-inline (%swig-pointer? x) (##core#inline "C_swigpointerp" x))

(define-inline (%locative? x) (##core#inline "C_locativep" x))

(define-inline (%random-fixnum x) (##core#inline "C_random_fixnum" x))

;;; Support code

(define-syntax (define-unique-object x r c)
  `(,(r 'define) ,(cadr x) (,(r 'gensym)) ))

(define (filter-in f l)
  (let loop ([l l])
    (cond [(null? l) '()]
          [else
            (let ([h (%car l )]
                  [r (%cdr l)] )
              (if (f h)
                  ; Unlike SRFI-1 filter this doesn't share the undeleted longest tail
                  #;(let ([t (loop r)]) (if (eq? r t) l (cons h t)))
                  (cons h (loop r))
                  (loop r) ) ) ] ) ) )

(define fast-getl
  (foreign-lambda* scheme-object ((scheme-object initargs) (scheme-object name) (scheme-object def)) "
    while(initargs != C_SCHEME_END_OF_LIST) {
      if(name == C_block_item(initargs, 0)) {
        if((initargs = C_block_item(initargs, 1)) == C_SCHEME_END_OF_LIST) return(def);
        else return(C_block_item(initargs, 0));
      }
      initargs = C_block_item(initargs, 1);
    }
    return(def);") )

(define-unique-object not-found-object)

(define (getl initargs name . def)
  (let ([value (fast-getl initargs name not-found-object)])
    (if (eq? value not-found-object)
        (optional def (##sys#error 'getl "cannot find item" name initargs))
        value ) ) )

;
; A simple topological sort.
;
; This is a fairly modified version of code I originally got from Anurag
; Mendhekar <anurag@moose.cs.indiana.edu>.
;

(define (compute-std-cpl c get-direct-supers)

  (define-inline (every1 test lst)
    (let loop ([lst lst])
      (or (null? lst)
          (and (test (%car lst))
               (loop (%cdr lst)) ) ) ) )

  (define (top-sort elements constraints tie-breaker)
    (let loop ((elements    elements)
               (constraints constraints)
               (result      '()))
        (if (null? elements)
            result
            (let ((can-go-in-now
                    (filter-in
                      (lambda (x)
                        (every1 (lambda (constraint)
                                 (or (not (eq? (%cadr constraint) x))
                                     (memq (%car constraint) result)))
                               constraints))
                      elements)))
              (if (null? can-go-in-now)
                  (##sys#error 'top-sort "invalid constraints")
                  (let ((choice (if (null? (%cdr can-go-in-now))
                                    (%car can-go-in-now)
                                    (tie-breaker result can-go-in-now))))
                    (loop (filter-in (lambda (x) (not (eq? x choice))) elements)
                          ; Include all constraints
                          #;(filter-in (lambda (x) (not (eq? (%cadr x) choice))) constraints)
                          constraints
                          (append result (list choice))))))) ) )

  (define (std-tie-breaker get-supers)
    (lambda (partial-cpl min-elts)
      (let loop ((pcpl (reverse partial-cpl)))
        (let* ((current-elt (%car pcpl))
              (ds-of-ce (get-supers current-elt))
              (common (filter-in (cut memq <> ds-of-ce) min-elts)))
          (if (null? common)
              (let ([r (%cdr pcpl)])
                (if (null? r)
                    (##sys#error 'std-tie-breaker "nothing valid")
                    (loop r)) )
              (%car common)) ) ) ) )

  (define (build-transitive-closure get-follow-ons)
    (lambda (x)
      (let track ((result '())
                  (pending (list x)))
           (if (null? pending)
               result
               (let ((next (%car pending)))
                 (if (memq next result)
                     (track result (%cdr pending))
                     (track (cons next result)
                            (append (get-follow-ons next)
                                    (%cdr pending)))))) ) ) )

  (define (build-constraints get-follow-ons)
    (lambda (x)
      (let loop ((elements ((build-transitive-closure get-follow-ons) x))
                 (this-one '())
                 (result '()))
           (if (or (null? this-one) (null? (%cdr this-one)))
               (if (null? elements)
                   result
                   (loop (%cdr elements)
                         (cons (%car elements)
                               (get-follow-ons (%car elements)))
                         result))
               (loop elements
                     (%cdr this-one)
                     (cons (list (%car this-one) (%cadr this-one))
                           result))) ) ) )

  (top-sort ((build-transitive-closure get-direct-supers) c)
            ((build-constraints get-direct-supers) c)
            (std-tie-breaker get-direct-supers) ) )

;;; Method cache support code:

#>
#define C_METHOD_CACHE_SIZE 8
<#

(define-foreign-variable method-cache-size int "C_METHOD_CACHE_SIZE")

(define method-caching-enabled #f)
(define method-cache-tag #f)

(define (make-method-cache)
  (cons method-cache-tag (make-vector (arithmetic-shift method-cache-size 1) #f)) )

(define method-cache-lookup
  (foreign-lambda* scheme-object ((scheme-object mcache) (scheme-object hash) (scheme-object classes)) "
    C_word v = C_block_item(mcache, 1);
    C_word clist, x, y;
    int free_index = -1;
    int i = ((C_unfix(hash) & (C_METHOD_CACHE_SIZE - 1)) << 1) & 0xffff,
        i0, i2;
    for(i0 = i;; i = i2) {
      clist = C_block_item(v, i);
      if(clist != C_SCHEME_FALSE) {
        x = classes;
        y = clist;
        while(x != C_SCHEME_END_OF_LIST && y != C_SCHEME_END_OF_LIST) {
          if(C_block_item(x, 0) != C_block_item(y, 0)) goto mismatch;
          else {
            x = C_block_item(x, 1);
            y = C_block_item(y, 1);
          }
        }
        if (x == C_SCHEME_END_OF_LIST && y == C_SCHEME_END_OF_LIST)
          return(C_block_item(v, i + 1));
        else
          goto mismatch;
      }
      else if(free_index == -1) free_index = i;
    mismatch:
      i2 = (i + 2) & ((C_METHOD_CACHE_SIZE << 1) - 1);
      if(i2 == i0) return(free_index == -1 ? C_SCHEME_FALSE : C_fix(free_index));
    }") )

;
; Then, we need to build what, in a more real implementation, would be
; the interface to the memory subsystem: instances and entities.  The
; former are used for instances of instances of <class>; the latter
; are used for instances of instances of <entity-class>.  In this MOP,
; none of this is visible to base- or MOP-level programmers.
;
; (One might consider rewriting the definition of instances using
; define-record or something.  It doesn't turn out to make it much
; simpler, at least not to me.  It also breaks the nice parallelism
; with entities.)
;
; %allocate-instance, %allocate-entity, get-field, set-field!, and class-of are
; the normal interface, from the rest of the code, to the low-level memory
; system.
;
; One thing to take note of is that the protocol does not allow the user
; to add low-level instance representations.  I have never seen a way to make
; that work.
;

; Instance

(define the-slots-of-a-class            ;
  '(direct-supers                       ;(class ...)
    direct-slots                        ;((name . options) ...)
    cpl                                 ;(class ...)
    slots                               ;((name . options) ...)
    nfields                             ;an integer
    field-initializers                  ;(proc ...)
    getters-n-setters                   ;((slot-name getter . setter) ...)
    name) )                             ;name

; structure: tag class cache direct-supers ... name
(define-constant basic-class-instance-size 11)

(define (%allocate-instance class nfields)
  (let ((instance (make-vector (+ nfields 3) #f)))
    (##core#inline "C_vector_to_structure" instance)
    (%structure-set! instance 0 'instance)
    (%structure-set! instance 1 class)
    instance))

(define-inline (%instance? x)
  (%structure? x 'instance))

(define-inline (%instance-class instance)
  (%structure-ref instance 1))

(define-inline (%set-instance-class! instance new-value)
  (%structure-set! instance 1 new-value))

(define-inline (%instance-cache-ref instance)
  (%structure-ref instance 2))

(define-inline (%instance-cache-set! instance x)
  (%structure-set! instance 2 x))

(define-inline (%instance-ref instance index)
  (%structure-ref instance (+ index 3)))

(define-inline (%instance-set! instance index new-value)
  (%structure-set! instance (+ index 3) new-value))

(define-record-printer (instance x out)
  (print-object x out) )

; Entity

(define-unique-object entity-tag)

(define %allocate-entity
  (let ([default-proc
	  (lambda args
	    (##sys#error '%allocate-entity "called entity without first setting proc"))] )
    (lambda (class nfields name)
      (letrec ((entity-def (make-vector (+ nfields 5) #f))
               (closure (lambda args (apply (%structure-ref entity-def 1) args))))
        (##core#inline "C_vector_to_structure" entity-def)
        (%structure-set! entity-def 0 'entity)
        (%structure-set! entity-def 1 default-proc)
        (%structure-set! entity-def 2 class)
        (%structure-set! entity-def 3 name)
        ;; slot #4 is cache (defaults to #f)
        (let ([len (%closure-length closure)])
          (let ([entity (make-vector (+ len 2))] )
            (do ([i 1 (add1 i)])
                ((>= i len)
                 (%vector-set! entity i entity-def)
                 (%vector-set! entity (add1 i) entity-tag)
                 (##core#inline "C_vector_to_closure" entity)
                 (##core#inline "C_copy_pointer" closure entity)
                 entity)
              (%vector-set! entity i (%closure-ref closure i)) ) ) ) ) ) ) )

(define-inline (%entity? x)
  (and (procedure? x)
       (let ([len (%closure-length x)])
         (and (> len 3)
              (eq? entity-tag (%closure-ref x (sub1 len))) ) ) ) )

(define-inline (%entity-def entity)
  (%closure-ref entity (- (%closure-length entity) 2)) )

(define-inline (%set-entity-proc! entity proc)
  (%structure-set! (%entity-def entity) 1 proc) )

(define-inline (%entity-class entity)
  (%structure-ref (%entity-def entity) 2) )

(define-inline (%entity-name entity)
  (%structure-ref (%entity-def entity) 3) )

(define-inline (%set-entity-name! entity x)
  (%structure-set! (%entity-def entity) 3 x) )

(define-inline (%entity-cache-ref entity)
  (%structure-ref (%entity-def entity) 4) )

(define-inline (%entity-cache-set! entity x)
  (%structure-set! (%entity-def entity) 4 x) )

(define-inline (%entity-ref entity index)
  (%structure-ref (%entity-def entity) (+ index 5)) )

(define-inline (%entity-set! entity index new-value)
  (%structure-set! (%entity-def entity) (+ index 5) new-value) )

(define-record-printer (entity x out)
  (print-object x out) )

; Instance/Entity field accessors

(define get-field
  (lambda (object field)
    (cond ((%instance? object) (%instance-ref object field))
          ((%entity?   object) (%entity-ref   object field))
          (else
           (##sys#signal-hook #:type-error 'get-field "can only get-field of instances and entities" object)))))

(define set-field!
  (lambda (object field new-value)
    (cond ((%instance? object) (%instance-set! object field new-value))
          ((%entity?   object) (%entity-set!   object field new-value))
          (else
           (##sys#signal-hook #:type-error 'set-field! "can only set-field! of instances and entities" object)))))

;
; Note that this implementation of class-of assumes the name of the
; primitive classes that are set up later.
;

; Local namespace - all "exported" procedures are inline so
; defined in compilation unit top-level scope.

(define (delete1! test lst)
    (let loop ((cpair lst) (ppair #f))
      (cond ((null? cpair) lst )
            ((test (%car cpair))
              (if ppair
                  (begin (%set-cdr! ppair (%cdr cpair)) lst)
                  (%cdr cpair)) )
            (else (loop (%cdr cpair) cpair) ) ) ) )

(define (any1 test lst)
    (let loop ([lst lst])
      (and (not (null? lst))
           (or (test (%car lst))
               (loop (%cdr lst)) ) ) ) )

  ; Class-of extension helpers
  ;
  ; Implemented as a map: "test" <-> "class"
  ;
  ; tst - test value; symbol or procedure
  ; cls - class
  ; prd - predicate? obj
  ; eql - symbol eq? obj

  (define-inline (clsmapelm-tst cme)
    (%car cme) )

  (define-inline (clsmapelm-cls cme)
    (%cdr cme) )

  (define-inline (clsmapelm-tst-set! cme tst)
    (%set-car! cme tst) )

  (define-inline (clsmapelm-cls-set! cme cls)
    (%set-cdr! cme cls) )

  (define-inline (clsmap-tst? tst)
    (or (procedure? tst) (symbol? tst)) )

  (define-inline (clsmapelm-cls-of/prd? cme tst)
    ((clsmapelm-tst cme) tst) )

  (define-inline (clsmapelm-cls-of/eql? cme tst)
    (eq? tst (clsmapelm-tst cme)) )

  (define-inline (clsmapelm-cls-of/prd cme tst)
    (and (clsmapelm-cls-of/prd? cme tst)
         (clsmapelm-cls cme)) )

  (define-inline (clsmapelm-cls-of/eql cme tst)
    (and (clsmapelm-cls-of/eql? cme tst)
         (clsmapelm-cls cme)) )

  (define-inline (clsmap-add cm tst cls)
    (cons (cons tst cls) cm) )

   (define clsmap-update
        (lambda (cm tst cls)
          (let ((cme (any1 (cut clsmapelm-cls-of/eql? <> tst) cm)))
            (if cme
                (begin (clsmapelm-tst-set! cme tst) (clsmapelm-cls-set! cme cls) cm)
                (clsmap-add cm tst cls) ) ) ))

      (define clsmap-cls-of/prd
        (lambda (cm tst)
          (any1 (cut clsmapelm-cls-of/prd <> tst) cm) ))

      (define clsmap-cls-of/eql
        (lambda (cm tst)
          (any1 (cut clsmapelm-cls-of/eql <> tst) cm) ))

      (define clsmap-del/tst
        (lambda (cm tst)
          (delete1! (cut clsmapelm-cls-of/eql? <> tst) cm) ))

      (define clsmap-del/cls
        (lambda (cm cls)
          (delete1! (lambda (cme) (eq? cls (clsmapelm-cls cme))) cm) ))

    ; Primitive class-of extensions

    (define *primitive-class-map* '())

      (define-inline (primitive-class-of x)
        (or (clsmap-cls-of/prd *primitive-class-map* x)
            #;<object>) )

      (define-inline (delete-primitive-class-of cls-or-tst)
        (set! *primitive-class-map*
              (if (clsmap-tst? cls-or-tst)
                  (clsmap-del/tst *primitive-class-map* cls-or-tst)
                  (clsmap-del/cls *primitive-class-map* cls-or-tst))) )

      (define-inline (update-primitive-class-of prd cls)
        (set! *primitive-class-map* (clsmap-update *primitive-class-map* prd cls)) )

    ; Structure class-of extensions

    (define *structure-class-map* '())

      (define-inline (structure-class-of x)
        (or (clsmap-cls-of/eql *structure-class-map* x)
            <structure>) )

      (define-inline (delete-structure-class-of cls-or-tst)
        (set! *structure-class-map*
              (if (clsmap-tst? cls-or-tst)
                  (clsmap-del/tst *structure-class-map* cls-or-tst)
                  (clsmap-del/cls *structure-class-map* cls-or-tst))) )

      (define-inline (update-structure-class-of tag cls)
        (set! *structure-class-map* (clsmap-update *structure-class-map* tag cls)) )

    ; Tagged-pointer class-of extensions

    (define *tagged-pointer-class-map* '())

      (define-inline (tagged-pointer-class-of x)
        (or (clsmap-cls-of/eql *tagged-pointer-class-map* (%tagged-pointer-data x))
            <tagged-pointer>) )

      (define-inline (delete-tagged-pointer-class-of cls-or-tst)
        (set! *tagged-pointer-class-map*
              (if (clsmap-tst? cls-or-tst)
                  (clsmap-del/tst *tagged-pointer-class-map* cls-or-tst)
                  (clsmap-del/cls *tagged-pointer-class-map* cls-or-tst))) )

      (define-inline (update-tagged-pointer-class-of tag cls)
        (set! *tagged-pointer-class-map* (clsmap-update *tagged-pointer-class-map* tag cls)) )

    ; Extended-procedure class-of extensions

    (define *extended-procedure-class-map* '())
    (define xproc-tag (vector 'extended))

      (define-inline (extended-procedure-lambda-decoration x)
        (##sys#lambda-decoration x (lambda (x) (and (pair? x) (equal? xproc-tag (%car x))))) )

      (define-inline (procedure-class-of x)
        (or (and-let* ([d (extended-procedure-lambda-decoration x)])
              (clsmap-cls-of/prd *extended-procedure-class-map* (%cdr d)) )
            <procedure>) )

      (define-inline (delete-extended-procedure-class-of cls-or-tst)
        (set! *extended-procedure-class-map*
              (if (clsmap-tst? cls-or-tst)
                  (clsmap-del/tst *extended-procedure-class-map* cls-or-tst)
                  (clsmap-del/cls *extended-procedure-class-map* cls-or-tst))) )

      (define-inline (update-extended-procedure-class-of prd cls)
        (set! *extended-procedure-class-map* (clsmap-update *extended-procedure-class-map* prd cls)) )

;

(define (class-of x)
  (cond [(%unbound? x)                  (##sys#error 'class-of "unbound object")]
        [(null? x)                      <null>]
        [(fixnum? x)                    <exact>]
        [(boolean? x)                   <boolean>]
        [(char? x)                      <char>]
        [(eof-object? x)                <end-of-file>]
        [(%undefined? x)                <void>]
        [(%immediate-value? x)          (##sys#error 'class-of "unidentified immediate object - cannot infer class" x)]
        [(flonum? x)                    <inexact>]
        [(integer? x)                   <integer>]
        [(symbol? x)                    <symbol>]
        [(%instance? x)                 (%instance-class x)]
        [(%entity? x)                   (%entity-class x)]
        [(vector? x)                    <vector>]
        [(pair? x)                      <pair>]
        [(string? x)                    <string>]
        [(procedure? x)                 (procedure-class-of x)]
        [(port? x)                      (if (input-port? x) <input-port> <output-port>)]
        [(%blob? x)                     <blob>]
        [(%pointer? x)                  <pointer>]
        [(%tagged-pointer? x)           (tagged-pointer-class-of x)]
        [(%swig-pointer? x)             <swig-pointer>]
        [(%locative? x)                 <locative>]
        [(%structure? x)
         (case (%structure-tag x)
           [(environment)               <environment>]
           [(array)                     <array>]
           [(hash-table)                <hash-table>]
           [(queue)                     <queue>]
           [(condition)                 <condition>]
           [(condition-variable)        <condition-variable>]
           [(char-set)                  <char-set>]
           [(time)                      <time>]
           [(lock)                      <lock>]
           [(mmap)                      <mmap>]
           [(promise)                   <promise>]
           [(u8vector)                  <u8vector>]
           [(s8vector)                  <s8vector>]
           [(u16vector)                 <u16vector>]
           [(s16vector)                 <s16vector>]
           [(u32vector)                 <u32vector>]
           [(s32vector)                 <s32vector>]
           [(f32vector)                 <f32vector>]
           [(f64vector)                 <f64vector>]
           [(tcp-listener)              <tcp-listener>]
           [(thread)                    <thread>]
           [(mutex)                     <mutex>]
           [(continuation)              <continuation>]
           [(read-table)                <read-table>]
           [(regexp)                    <regexp>]
           [else                        (structure-class-of x)] ) ]
        [(primitive-class-of x)]
        [else
          (##sys#error 'class-of "unidentified primitive object - cannot infer class" x) ] ) )

;
; Now we can get down to business.  First, we initialize the braid.
;
; For Bootstrapping, we define an early version of MAKE.  It will be
; changed to the real version later on.  String search for ``set! make''.
;

(randomize)

(define (make class . initargs)
  (cond ((or (eq? class <class>)
             (eq? class <entity-class>))
         (let* ((new (%allocate-instance class (length the-slots-of-a-class)))
                (dsupers (fast-getl initargs 'direct-supers '()))
                (name (fast-getl initargs 'name "(anonymous)"))
                (dslots  (map list (fast-getl initargs 'direct-slots  '())))
                (cpl     (let loop ((sups dsupers) (so-far (list new)))
                           (if (null? sups)
                               (reverse so-far)
                               (let ([cls (%car sups)])
                                 (loop (class-direct-supers cls) (cons cls so-far))))))
                (slots (apply append dslots (map class-direct-slots (%cdr cpl))))
                (nfields 0)
                (field-initializers '())
                (allocator
                 (lambda (init)
                   (let ((f nfields))
                     (set! nfields (add1 nfields))
                     (set! field-initializers
                       (cons init field-initializers))
                     (values (lambda (o)   (get-field  o f))
                             (lambda (o n) (set-field! o f n))))))
                (getters-n-setters
                 (map (lambda (s)
                        (cons (%car s)
                              (call-with-values (lambda () (allocator (lambda () (void)))) cons) ) )
                      slots)))
           (##tinyclos#slot-set! new 'direct-supers      dsupers)
           (##tinyclos#slot-set! new 'direct-slots       dslots)
           (##tinyclos#slot-set! new 'cpl                cpl)
           (##tinyclos#slot-set! new 'slots              slots)
           (##tinyclos#slot-set! new 'nfields            nfields)
           (##tinyclos#slot-set! new 'field-initializers (reverse field-initializers))
           (##tinyclos#slot-set! new 'getters-n-setters  getters-n-setters)
           (##tinyclos#slot-set! new 'name               name)
           (%instance-cache-set! new (%random-fixnum #x10000))
           new))
        ((eq? class <generic>)
         (let ([new (%allocate-entity class (length (class-slots class))
                                      (fast-getl initargs 'name "(unnamed)") ) ] )
           (##tinyclos#slot-set! new 'methods '())
           new))
        ((eq? class <method>)
         (let ((new (%allocate-instance class (length (class-slots class)))))
           (##tinyclos#slot-set! new 'specializers (getl initargs 'specializers))
           (##tinyclos#slot-set! new 'procedure    (getl initargs 'procedure))
           new))
        (else
          (##sys#error "bootstrap make: unknown class" class)) ) )

;
; These are the real versions of slot-ref and slot-set!.  Because of the
; way the new slot access protocol works, with no generic call inline,
; they can be defined up front like this.  Cool eh?
;

(define (##tinyclos#slot-ref object slot-name)
  ; if true, then this is an instance of <class>, with no additional slots
  (if (and (%instance? object) (%instance-cache-ref object))
      (%instance-ref
       object
       (case slot-name
         [(direct-supers)       0]
         [(direct-slots)        1]
         [(cpl)                 2]
         [(slots)               3]
         [(nfields)             4]
         [(field-initializers)  5]
         [(getters-n-setters)   6]
         [(name)                7]
         [else (##sys#error "unknown basic slot-name" slot-name)] ) )
      (let* ((info   (lookup-slot-info (class-of object) slot-name))
             (getter (%car info)))
        (getter object))) )

(define (##tinyclos#slot-set! object slot-name new-value)
  (let* ((info   (lookup-slot-info (class-of object) slot-name))
         (setter (%cdr info)) )
    (setter object new-value)) )

(define slot-ref (getter-with-setter ##tinyclos#slot-ref ##tinyclos#slot-set!))
(define slot-set! ##tinyclos#slot-set!)

(define (lookup-slot-info class slot-name)
  (let* ((getters-n-setters
          (if (eq? class <class>)         ;* This grounds out
              getters-n-setters-for-class ;* the slot-ref tower.
              (##tinyclos#slot-ref class 'getters-n-setters)))
         (entry (assq slot-name getters-n-setters)))
    (if entry
        (%cdr entry)
        (##sys#error "no slot in instances of class" slot-name class) ) ) )

;
; Given that the early version of MAKE is allowed to call accessors on
; class metaobjects, the definitions for them come here, before the
; actual class definitions, which are coming up right afterwards.
;

(define (class-direct-slots class)
  (##tinyclos#slot-ref class 'direct-slots) )
(define (class-direct-supers class)
  (##tinyclos#slot-ref class 'direct-supers))
(define (class-slots class)
  (##tinyclos#slot-ref class 'slots) )

(define (class-name class)
  (##tinyclos#slot-ref class 'name))

(define (generic-methods generic)
  (##tinyclos#slot-ref generic 'methods) )

(define (method-specializers method)
  (##tinyclos#slot-ref method 'specializers) )
(define (method-procedure method)
  (##tinyclos#slot-ref method 'procedure) )

(define (class-cpl class)
  (##tinyclos#slot-ref class 'cpl) )

;;; Inline procedures inside this module only:

(eval-when (compile)
  (define-syntax define-inline-accessor
    (syntax-rules ()
      ((_ name slot)
       (define-syntax name
	 (syntax-rules ()
	   ((_ class) (slot-ref class 'slot)))))))
  (define-inline-accessor class-direct-slots direct-slots)
  (define-inline-accessor class-direct-supers direct-supers)
  (define-inline-accessor class-slots slots)
  (define-inline-accessor class-name name)
  (define-inline-accessor generic-methods methods)
  (define-inline-accessor method-specializers specializers)
  (define-inline-accessor method-procedure procedure)
  (define-inline-accessor class-cpl cpl))

;
; The next 7 clusters define the 6 initial classes.  It takes 7 to 6
; because the first and fourth both contribute to <class>.
;

(define getters-n-setters-for-class       ;see lookup-slot-info
  (let loop ([lst the-slots-of-a-class] [i 0])
    (if (null? lst)
        '()
        (cons (cons (%car lst)
                    (cons (lambda (o)   (%instance-ref  o i))
                          (lambda (o n) (%instance-set! o i n)) ) )
              (loop (%cdr lst) (add1 i)) ) ) ) )

(define <class> (%allocate-instance #f (length the-slots-of-a-class)))
(%set-instance-class! <class> <class>)

(define <top>          (make <class>
                             'direct-supers '()
                             'direct-slots  '()
                             'name          "top"))

(define <object>       (make <class>
                             'direct-supers (list <top>)
                             'direct-slots  '()
                             'name          "object"))

;
; This cluster, together with the first cluster above that defines
; <class> and sets its class, have the effect of:
;
;   (define <class>
;     (make <class>
;           'direct-supers (list <object>)
;           'direct-slots  (list 'direct-supers ...)))
;

(%instance-set! <class> 0 (list <object>))                  ;d supers
(%instance-set! <class> 1 (map list the-slots-of-a-class))  ;d slots
(%instance-set! <class> 2 (list <class> <object> <top>))    ;cpl
(%instance-set! <class> 3 (map list the-slots-of-a-class))  ;slots
(%instance-set! <class> 4 (length the-slots-of-a-class))    ;nfields
(%instance-set! <class> 5 (map (lambda (s)                  ;field-ini..
                                 (lambda () (void)))
                               the-slots-of-a-class))
(%instance-set! <class> 6 '())
(%instance-set! <class> 7 'class)
(%instance-cache-set! <class> (%random-fixnum #x10000))

(define <procedure-class> (make <class>
                                'direct-supers (list <class>)
                                'direct-slots  '()
                                'name          "procedure-class"))

(define <entity-class>    (make <class>
                                'direct-supers (list <procedure-class>)
                                'direct-slots  '()
                                'name          "entity-class"))

(define <generic>         (make <entity-class>
                                'direct-supers (list <object>)
                                'direct-slots  (list 'methods)
                                'name          "generic"))

(define <method>          (make <class>
                                'direct-supers (list <object>)
                                'direct-slots  (list 'specializers 'procedure)
                                'name          "method"))

;
; These are the convenient syntax we expose to the base-level user.
;

(define (make-class direct-supers direct-slots . name)
  (make <class>
        'direct-supers direct-supers
        'direct-slots  direct-slots
        'name          (optional name "(anonymous)")) )

(define (make-generic . name)
  (make <generic>
        'name         (optional name "(unnamed)")) )

(define (make-method specializers procedure)
  (make <method>
        'specializers specializers
        'procedure    procedure) )

;
; The initialization protocol
;

(define initialize (make-generic "initialize"))

;
; The instance structure protocol.
;

(define allocate-instance (make-generic "allocate-instance"))
(define compute-getter-and-setter (make-generic "compute-getter-and-setter"))

;
; The class initialization protocol.
;

(define compute-cpl (make-generic "compute-cpl"))
(define compute-slots (make-generic "compute-slots"))

;
; The generic invocation protocol.
;

(define compute-apply-generic         (make-generic "compute-apply-generic"))
(define compute-methods               (make-generic "compute-methods"))
(define compute-method-more-specific? (make-generic "compute-method-more-specific?"))
(define compute-apply-methods         (make-generic "compute-apply-methods"))

;
; The next thing to do is bootstrap generic functions.
;

(define generic-invocation-generics (list compute-apply-generic
                                          compute-methods
                                          compute-method-more-specific?
                                          compute-apply-methods))

(define (add-method generic method)
  (##tinyclos#slot-set!
    generic
    'methods
    (let* ([ms1 (method-specializers method)]
           [l1 (length ms1)] )
      (let filter-in-method ([methods (##tinyclos#slot-ref generic 'methods)])
        (if (null? methods)
            (list method)
            (let* ([mm (%car methods)]
                   [ms2 (method-specializers mm)]
                   [l2 (length ms2)])
              (cond ((> l1 l2)
                     (cons mm (filter-in-method (%cdr methods))))
                    ((< l1 l2)
                     (cons method methods))
                    (else
                     (let check-method ([ms1 ms1]
                                        [ms2 ms2])
                       (cond ((and (null? ms1) (null? ms2))
                              (cons method (%cdr methods))) ;; skip the method already in the generic
                             ((eq? (%car ms1) (%car ms2))
                              (check-method (%cdr ms1) (%cdr ms2)))
                             (else
                              (cons mm (filter-in-method (%cdr methods)))))))))))))
    (if (memq generic generic-invocation-generics)
        (set! method-cache-tag (vector))
        (%entity-cache-set! generic #f) )
    (%set-entity-proc! generic (compute-apply-generic generic)) )

;
; Adding a method calls COMPUTE-APPLY-GENERIC, the result of which calls
; the other generics in the generic invocation protocol.  Two, related,
; problems come up.  A chicken and egg problem and a infinite regress
; problem.
;
; In order to add our first method to COMPUTE-APPLY-GENERIC, we need
; something sitting there, so it can be called.  The first definition
; below does that.
;
; Then, the second definition solves both the infinite regress and the
; not having enough of the protocol around to build itself problem the
; same way: it special cases invocation of generics in the invocation
; protocol.
;
;

(%set-entity-proc! compute-apply-generic
     (lambda (generic)             ;The ONE time this is called
                                   ;it doesn't get cnm.
       (lambda args
         (apply (method-procedure (car (generic-methods generic)))
                #f args)))) ;But, the ONE time it is run,
                            ;it needs to pass a dummy
                            ;value for cnm!

(let ([symbol-vector (vector 'instance entity-tag)])

  ; Compute class ID from object
  (define hash-arg-list
    (foreign-lambda* unsigned-int ((scheme-object args) (scheme-object svector)) "
      C_word tag, h, x;
      int n, i, j, len = 0;
      for(i = 0; args != C_SCHEME_END_OF_LIST; args = C_block_item(args, 1)) {
        x = C_block_item(args, 0);
        if(C_immediatep(x)) {
          switch(x) {
            case C_SCHEME_END_OF_LIST: i += 1; break;
            case C_SCHEME_TRUE:
            case C_SCHEME_FALSE: i += 3; break;
            case C_SCHEME_END_OF_FILE: i += 7; break;
            case C_SCHEME_UNDEFINED: i += 5; break;
            default:
              if(x & C_FIXNUM_BIT) i += 2;
              else i += 4;
          }
        }
        else {
          h = C_header_bits(x);
          switch(h) {
          case C_STRUCTURE_TYPE:
            tag = C_block_item(x, 0);
            if(tag == C_block_item(svector, 0)) { /* instance */
              if((tag = C_block_item(C_block_item(x, 1), 2)) != C_SCHEME_FALSE) i += C_unfix(tag);
              else i += C_header_size(x) << 4;
            }
            else i += 17;
            break;
          case C_CLOSURE_TYPE:
            n = C_header_size(x);
            if(n > 3 && C_block_item(svector, 1) == C_block_item(x, n - 1)) {
              if((tag = C_block_item(C_block_item(C_block_item(x, n - 2), 2), 2)) != C_SCHEME_FALSE) i += C_unfix(tag);
              else i += 13;
            }
            break;
          case C_SYMBOL_TYPE: i += 8; break;
          case C_BYTEVECTOR_TYPE: i += 16; break;
          case C_VECTOR_TYPE: i += 9; break;
          case C_PAIR_TYPE: i += 10; break;
          case C_FLONUM_TYPE: i += 11; break;
          case C_STRING_TYPE: i += 12; break;
          case C_PORT_TYPE: i += C_block_item(x, 1) ? 15 : 14; break;
          default: i += 255;
          }
        }
        ++len;
      }
      return((i + len) & (C_METHOD_CACHE_SIZE - 1));") )

  (add-method compute-apply-generic
    (make-method (list <generic>)
      (lambda (call-next-method generic)
        (lambda args
          (let ([mc (%entity-cache-ref generic)])
            (when (or (not mc) (not (eq? method-cache-tag (%car mc))))
              (set! mc (make-method-cache))
              (%entity-cache-set! generic mc) )
            (let* ([classes (and method-caching-enabled (map class-of args))]
                   [key (and classes (hash-arg-list args symbol-vector))]
                   [e (and classes (method-cache-lookup mc key classes))] )
              ;(unless (##sys#immediate? e) (print (%entity-name generic) ": " key))
              (if (not (##sys#immediate? e))
                  (e args)
                  (let ([cam
                         (if (and (memq generic generic-invocation-generics)
                                  (memq (car args) generic-invocation-generics))
                             (let ([proc
                                    (method-procedure
                                      ; select the first method of one argument
                                     (let loop ([lis (generic-methods generic)])
                                       (if (null? lis)
                                         (##sys#error "cannot find original compute-apply-generic")
                                         (let* ([h (%car lis)]
                                                [ms (method-specializers h)])
                                           (if (= 1 (length ms))
                                               h
                                               (loop (%cdr lis))))))) ] )
                               (lambda (args) (apply proc #f args)) )
                             (let ([x (compute-apply-methods generic)]
                                   [y ((compute-methods generic) args)] )
                               (lambda (args) (x y args)) ) ) ] )
                    (when (and e method-caching-enabled)
                      (let ([v (%cdr mc)])
                        (%vector-set! v e classes)
                        (%vector-set! v (add1 e) cam) ) )
                    (cam args) ) ) ) ) ) ) ) ) )

(add-method compute-methods
  (make-method (list <generic>)
    (lambda (call-next-method generic)
      (lambda (args)
        (let ([applicable
               (filter-in (lambda (method)
                            (let check-applicable ([lst1 (method-specializers method)]
                                                   [lst2 args])
                              (cond ((null? lst1) #t)
                                    ((null? lst2) #f)
                                    (else
                                      (and (applicable? (%car lst1) (%car lst2))
                                           (check-applicable (%cdr lst1) (%cdr lst2)))))))
                          (generic-methods generic) ) ] )
          (if (or (null? applicable) (null? (%cdr applicable)))
              applicable
              (let ([cmms (compute-method-more-specific? generic)])
                (sort applicable (lambda (m1 m2) (cmms m1 m2 args))) ) ) ) ) ) ) )

(add-method compute-method-more-specific?
  (make-method (list <generic>)
    (lambda (call-next-method generic)
      (lambda (m1 m2 args)
        (let loop ((specls1 (method-specializers m1))
                   (specls2 (method-specializers m2))
                   (args args))
          (cond-expand
           [unsafe
            (let ((c1  (%car specls1))
                  (c2  (%car specls2))
                  (arg (%car args)))
              (if (eq? c1 c2)
                  (loop (%cdr specls1)
                        (%cdr specls2)
                        (%cdr args))
                  (more-specific? c1 c2 arg))) ]
           [else
            (cond ((and (null? specls1) (null? specls2))
                   (##sys#error "two methods are equally specific" generic))
                  #; ; Ok to have diff # of specializers
                  ((or (null? specls1) (null? specls2))
                   (##sys#error "two methods have different number of specializers" generic))
                  ((null? specls1) #f)
                  ((null? specls2) #t)
                  ((null? args)
                   (##sys#error "fewer arguments than specializers" generic))
                  (else
                   (let ((c1  (%car specls1))
                         (c2  (%car specls2))
                         (arg (%car args)))
                     (if (eq? c1 c2)
                         (loop (%cdr specls1)
                               (%cdr specls2)
                               (%cdr args))
                         (more-specific? c1 c2 arg)))) ) ] ) ) ) ) ) )

(add-method compute-apply-methods
  (make-method (list <generic>)
    (lambda (call-next-method generic)
      (lambda (methods args)
        (letrec ((one-step
                  (lambda (tail)
                    (lambda ()
                      (cond-expand
                       [unsafe]
                       [else
                        (when (null? tail)
                          (##sys#error "call-next-method: no methods left" generic) ) ] )
                      (apply (method-procedure (%car tail))
                             (one-step (%cdr tail)) args)))))
          ((one-step methods)))))))

(define (applicable? c arg)
  (memq c (class-cpl (class-of arg))) )

(define (more-specific? c1 c2 arg)
  (memq c2 (memq c1 (class-cpl (class-of arg)))) )

(add-method initialize
  (make-method (list <top>)
    (lambda (call-next-method object initargs)
      (##sys#error "cannot initialize object" object) ) ) )

(add-method initialize
  (make-method (list <object>)
    (lambda (call-next-method object initargs) object)))

(add-method initialize
  (make-method (list <class>)
    (lambda (call-next-method class initargs)
      (call-next-method)
      (##tinyclos#slot-set! class 'direct-supers (fast-getl initargs 'direct-supers '()))
      (let ([dslots
              (map (lambda (s) (if (pair? s) s (list s)))
                   (fast-getl initargs 'direct-slots '()))])
        (let ([namestr
                (let ([name (fast-getl initargs 'name "(anonymous)")])
                  (cond [(symbol? name) (##sys#symbol->string name)]
                        [(string? name) name]
                        [else (##sys#signal-hook #:type-error
                                                 'initialize "invalid class name" name) ] ) ) ] )
          (##tinyclos#slot-set! class 'direct-slots dslots)
          (##tinyclos#slot-set! class 'cpl (compute-cpl class))
          (##tinyclos#slot-set! class 'name
                                      (let ([namestrlen (%string-length namestr)])
                                        (if (and (> namestrlen 0)
                                                 (char=? #\< (string-ref namestr 0))
                                                 (char=? #\> (string-ref namestr (sub1 namestrlen))) )
                                            (substring namestr 1 (sub1 namestrlen))
                                            namestr) ))
          (let ([slots (compute-slots class)])
            (##tinyclos#slot-set! class 'slots slots)
            (when (= basic-class-instance-size (%structure-length class))
              (%instance-cache-set! class (%random-fixnum #x10000)) )
            (let* ([nfields 0]
                   [field-initializers '()]
                   [allocator
                    (lambda (init)
                      (let ((f nfields))
                        (set! nfields (add1 nfields))
                        (set! field-initializers (cons init field-initializers))
                        (values (lambda (o)   (get-field  o f))
                                (lambda (o n) (set-field! o f n)))))]
                   [getters-n-setters
                    (map (lambda (slot)
                           (cons (%car slot)
                                 (call-with-values (lambda ()
                                                     (compute-getter-and-setter class slot allocator))
                                                   cons) ) )
                         slots) ] )
              (##tinyclos#slot-set! class 'nfields nfields)
              (##tinyclos#slot-set! class 'field-initializers (reverse field-initializers))
              (##tinyclos#slot-set! class 'getters-n-setters getters-n-setters))))) ) ) )

(add-method initialize
  (make-method (list <generic>)
    (lambda (call-next-method generic initargs)
      (call-next-method)
      (unless (%entity? generic)
        (##sys#error 'initialize "generic is not an entity") )
      (##tinyclos#slot-set! generic 'methods '())
      (%set-entity-name! generic (fast-getl initargs 'name "(unnamed)"))
      (%set-entity-proc! generic (lambda args (##sys#error "has no methods" generic))))))

(add-method initialize
  (make-method (list <method>)
    (lambda (call-next-method method initargs)
      (call-next-method)
      (##tinyclos#slot-set! method 'specializers (getl initargs 'specializers))
      (##tinyclos#slot-set! method 'procedure    (getl initargs 'procedure)))))

(add-method allocate-instance
  (make-method (list <class>)
    (lambda (call-next-method class)
      (let* ((field-initializers (##tinyclos#slot-ref class 'field-initializers))
             (new (%allocate-instance class (length field-initializers))))
        (let loop ((n 0) (inits field-initializers))
          (if (null? inits)
              new
              (begin
                (%instance-set! new n ((%car inits)))
                (loop (add1 n) (%cdr inits)))))))))

(add-method allocate-instance
  (make-method (list <entity-class>)
    (lambda (call-next-method class)
      (let* ([field-initializers (##tinyclos#slot-ref class 'field-initializers)]
             [new (%allocate-entity class (length field-initializers) "(unnamed)") ] )
        (let loop ((n 0) (inits field-initializers))
          (if (null? inits)
              new
              (begin
                (%entity-set! new n ((%car inits)))
                (loop (add1 n) (%cdr inits)))))))))

(add-method compute-cpl
    (make-method (list <class>)
      (lambda (call-next-method class)
        (compute-std-cpl class class-direct-supers))))

(add-method compute-slots
  (make-method (list <class>)
    (lambda (call-next-method class)
      (let collect ((to-process (apply append (map class-direct-slots (class-cpl class))))
                    (result '()))
        (if (null? to-process)
            (reverse result)
            (let* ((current (%car to-process))
                   (name (%car current))
                   (others '())
                   (remaining-to-process
                     (filter-in (lambda (o)
                                  (if (eq? (%car o) name)
                                      (begin
                                        (set! others (cons o others))
                                        #f)
                                      #t))
                                (%cdr to-process))))
              (collect remaining-to-process
                       (cons (append current (apply append (map (lambda (x) (%cdr x)) others)))
                             result))))))))

(add-method compute-getter-and-setter
  (make-method (list <class>)
    (lambda (call-next-method class slot allocator)
      (allocator (lambda () (void))))))

;
; Now everything works, both generic functions and classes, so we can
; turn on the real MAKE.
;

(set! make
  (lambda (class . initargs)
    (let ((instance (allocate-instance class)))
      (initialize instance initargs)
      instance)))

;
; Now define what CLOS calls `built in' classes.
;

(define <primitive-class>
  (make <class>
        'direct-supers (list <class>)
        'direct-slots  '()
        'name          "primitive-class"))

(define <primitive>
  (make <class>
        'direct-supers (list <top>)
        'direct-slots  '()
        'name          "primitive"))

(add-method initialize
  (make-method (list <primitive>)
    (lambda (call-next-method object initargs) object)))

(define (make-primitive-class name . sclasses)
 (make <primitive-class>
       'direct-supers (if (null? sclasses) (list <primitive>) sclasses)
       'direct-slots  '()
       'name          name) )

(define (make-port-class name)
  (make <class>
        'direct-supers (list <port>)
        'direct-slots  '()
        'name          name) )

(define (make-structure-class name)
  (make-primitive-class name <structure>) )

(define (make-extended-procedure-class name)
  (make-primitive-class name <procedure>) )

(define (make-tagged-pointer-class name)
  (make <class>
        'direct-supers (list <tagged-pointer>)
        'direct-slots  '()
        'name          name) )

(define <void>                  (make-primitive-class "void"))
(define <null>                  (make-primitive-class "null"))
(define <end-of-file>           (make-primitive-class "end-of-file"))
(define <boolean>               (make-primitive-class "boolean"))
(define <symbol>                (make-primitive-class "symbol"))
(define <char>                  (make-primitive-class "char"))
(define <number>                (make-primitive-class "number"))
(define <integer>               (make-primitive-class "integer" <number>))
(define <exact>                 (make-primitive-class "exact" <integer>))
(define <inexact>               (make-primitive-class "inexact" <number>))
(define <vector>                (make-primitive-class "vector"))
(define <pair>                  (make-primitive-class "pair"))
(define <string>                (make-primitive-class "string"))
(define <port>                  (make-primitive-class "port"))
(define <input-port>            (make-port-class "input-port"))
(define <output-port>           (make-port-class "output-port"))
(define <procedure>             (make-primitive-class "procedure" <procedure-class>))
(define <blob>                  (make-primitive-class "blob"))
(define <byte-vector>           (make-primitive-class "byte-vector")) ; DEPRECATED
(define <structure>             (make-primitive-class "structure"))
(define <environment>           (make-structure-class "environment"))
(define <hash-table>            (make-structure-class "hash-table"))
(define <promise>               (make-structure-class "promise"))
(define <queue>                 (make-structure-class "queue"))
(define <condition>             (make-structure-class "condition"))
(define <condition-variable>    (make-structure-class "condition-variable"))
(define <continuation>          (make-structure-class "continuation"))
(define <char-set>              (make-structure-class "char-set"))
(define <time>                  (make-structure-class "time"))
(define <lock>                  (make-structure-class "lock"))
(define <mmap>                  (make-structure-class "mmap"))
(define <array>                 (make-structure-class "array"))
(define <tcp-listener>          (make-structure-class "tcp-listener"))
(define <thread>                (make-structure-class "thread"))
(define <mutex>                 (make-structure-class "mutex"))
(define <regexp>                (make-structure-class "regexp"))
(define <read-table>            (make-structure-class "read-table"))
(define <locative>              (make-primitive-class "locative"))
(define <pointer>               (make-primitive-class "pointer"))
(define <swig-pointer>          (make-primitive-class "swig-pointer" <pointer>))
(define <tagged-pointer>        (make-primitive-class "tagged-pointer" <pointer>))
(define <u8vector>              (make-primitive-class "u8vector" <vector>))
(define <s8vector>              (make-primitive-class "s8vector" <vector>))
(define <u16vector>             (make-primitive-class "u16vector" <vector>))
(define <s16vector>             (make-primitive-class "s16vector" <vector>))
(define <u32vector>             (make-primitive-class "u32vector" <vector>))
(define <s32vector>             (make-primitive-class "s32vector" <vector>))
(define <f32vector>             (make-primitive-class "f32vector" <vector>))
(define <f64vector>             (make-primitive-class "f64vector" <vector>))

(define <c++-object>
  (make <class>
        'direct-supers (list <object>)
        'direct-slots  '(this)
        'name          "c++-object"))

(add-method initialize
  (make-method (list <c++-object>)
    (lambda (call-next-method obj initargs)
      (when (and (pair? initargs) (eq? 'this (%car initargs)))
        (##tinyclos#slot-set! obj 'this (cadr initargs)) ) ) ) )

(set! method-caching-enabled #t)

;;; Utilities:

(define (initialize-slots object initargs)
  (##sys#check-list initargs 'initialize-slots)
  (for-each
   (lambda (slot)
     (let* ([name   (car slot)]
            [value  (fast-getl initargs name not-found-object)] )
       (unless (eq? value not-found-object)
         (slot-set! object name value))))
   (class-slots (class-of object))) )

(define print-object (make-generic "print-object"))
(define describe-object (make-generic "describe-object"))

(add-method print-object
  (make-method (list <object>)
    (lambda (call-next-method x . port)
      (fprintf (optional port ##sys#standard-output) "#<~A>" (class-name (class-of x))) ) ) )

(add-method print-object
  (make-method (list <primitive>)
    (lambda (call-next-method x . port)
      (write x (optional port ##sys#standard-output)) ) ) )

(add-method print-object
  (make-method (list <class>)
    (lambda (call-next-method x . port)
      (fprintf (optional port ##sys#standard-output) "#<class ~A>" (class-name x)) ) ) )

(add-method print-object
  (make-method (list <generic>)
    (lambda (call-next-method x . port)
      (fprintf (optional port ##sys#standard-output) "#<generic ~A>" (%entity-name x)) ) ) )

(add-method describe-object
  (make-method (list <object>)
    (lambda (call-next-method x . port)
      (let ([class (class-of x)]
            [port (optional port ##sys#standard-output)] )
        (fprintf port "instance of class ~A:~%" (class-name class))
        (for-each
         (lambda (s)
           (let ([slot (car s)])
             (fprintf port " ~S\t-> ~S~%" slot (slot-ref x slot)) ) )
         (class-slots class) ) ) ) ) )

(add-method describe-object
  (make-method (list <class>)
    (lambda (call-next-method x . port)
      (fprintf (optional port ##sys#standard-output) "class ~A~%" (class-name x)) ) ) )

(add-method describe-object
  (make-method (list <generic>)
    (lambda (call-next-method x . port)
      (fprintf (optional port ##sys#standard-output) "generic ~A~%" (%entity-name x)) ) ) )

(define ensure-generic
  (let ([make-generic make-generic])
    (lambda (x sym)
      (if (%entity? x)
          x
          (make-generic (##sys#symbol->string sym)) ) ) ) )

(define add-global-method
  (let ([make-method make-method]
        [add-method add-method] )
    (lambda (val sym specializers proc)
      (let ([g (ensure-generic val sym)])
        (add-method g (make-method specializers proc))
        g) ) ) )

(define (instance? x)
  (or (%instance? x) (%entity? x)) )

(define (subclass? x y)
  (if (memq y (compute-cpl x)) #t #f) )

(define (instance-of? x class)
  (let ([cl (class-of x)])
    (or (eq? cl class)
        (subclass? cl class) ) ) )

(define (make-instance-from-pointer ptr cls)
  (and ptr
       (not (%null-pointer? ptr))
       (make cls 'this ptr) ) )

(define (make/copy x . initargs)
  (let ([class (class-of x)])
    (apply make class
                (let ([initlist initargs]
                      [inited-slot?
                        (lambda (nam)
                          (let loop ([testing? #t] [initargs initargs])
                            (and (pair? initargs)
                                 (or (and testing? (equal? nam (car initargs)))
                                     (loop (not testing?) (cdr initargs)) ) ) ) )])
                  (for-each
                    (lambda (s)
                      (let ([nam (car s)])
                        (unless (inited-slot? nam)
                          (set! initlist (cons nam (cons (slot-ref x nam) initlist))) ) ) )
                    (class-slots class))
                  initlist ) ) ) )

;; Procedural interface for extending the "builtin" class system

; Primitive

(define (new-primitive-class name pred . sclasses)
  (let ((cls (apply make-primitive-class name sclasses)))
    (update-primitive-class-of pred cls)
    cls ) )

(define (delete-primitive-class cls)
  (delete-primitive-class-of cls) )

; Structure

(define (new-structure-class name tag)
  (let ((cls (make-structure-class name)))
    (update-structure-class-of tag cls)
    cls ) )

(define (delete-structure-class cls)
  (delete-structure-class-of cls) )

; Tagged-pointer

(define (new-tagged-pointer-class name tag)
  (let ((cls (make-tagged-pointer-class name)))
    (update-tagged-pointer-class-of tag cls)
    cls ) )

(define (delete-tagged-pointer-class cls)
  (delete-tagged-pointer-class-of cls) )

; Procedure

(define (new-extended-procedure-class name pred)
  (let ((cls (make-extended-procedure-class name)))
    (update-extended-procedure-class-of pred cls)
    cls ) )

(define (delete-extended-procedure-class cls)
  (delete-extended-procedure-class-of cls) )

)
